/***************************************************************************
 *
 * Copyright (c) 2013 Codethink Limited
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/

#include <iostream>
#include <sstream>
#include <string>

#include "Log.h"

#include "CalibrationHelpers.h"
#include "Subdivision.h"
#include "ButtonSubdivision.h"

using namespace std;
using namespace LayerManagerCalibration;


bool ButtonSubdivision::validate()
{
    bool valid = true;

    // Verify that key code was parsed
    if (!m_bKeyCodeParsed)
    {
        LOG_WARNING("ButtonSubdivision", "No key code defined");
        valid = false;
    }

    // Verify that minimum repetition interval was parsed
    if (!m_bMinimumRepetitionIntervalParsed)
    {
        LOG_WARNING("ButtonSubdivision",
                    "No minimum repetition interval was defined");
        valid = false;
    }

    // Verify that the minimum repetition interval is positive
    if (m_fpMinimumRepetitionInterval < 0)
    {
        LOG_WARNING("ButtonSubdivision",
                    "Minimum repetition interval is negative");
        valid = false;
    }

    return valid;
}

bool ButtonSubdivision::handleUnknownKey(const string& key, const string& value)
{
    if (key.compare("key code") == 0)
    {
        istringstream iss(value);
        iss >> m_u32KeyCode;
        if (iss.fail())
        {
            // Failed to write the string to m_u32KeyCode.
            // It may be the symbolic name of the key.
            bool isKey;
            uint key = parseKeyCode(value, isKey);
            if (isKey)
            {
                m_u32KeyCode = key;
            }
            else
            {
                LOG_WARNING("ButtonSubdivision",
                            "Found invalid key code, value=" << value);
                return false;
            }
        }

        LOG_INFO("ButtonSubdivision",
                 "Found key code=" << m_u32KeyCode);
        m_bKeyCodeParsed = true;
    }
    else if (key.compare("minimum repetition interval") == 0)
    {
        istringstream iss(value);
        iss >> m_fpMinimumRepetitionInterval;
        if (iss.fail())
        {
            LOG_WARNING("ButtonSubdivision",
                        "Found invalid minimum repetition interval, "
                        "value=" << value);
            return false;
        }
        else
        {
            LOG_INFO("ButtonSubdivision",
                     "Found minimum repetition interval="
                     << m_fpMinimumRepetitionInterval);
            m_bMinimumRepetitionIntervalParsed = true;
        }
    }
    else
    {
        return false;
    }

    return true;
}
